unit IWRegion;

interface

uses
  Classes,
  {$IFDEF VSNET}
    System.ComponentModel,
    System.ComponentModel.Design,
    System.Drawing,
    System.Drawing.Design,

    IWNetBaseControl,
    IWNetBaseContainer,
    IWNetComponent,
    IWNetClasses,
  {$ELSE}
    {$IFDEF Linux}
      IWCLXBaseControl,
    {$ELSE}
      IWVCLBaseControl,
      IWVCLBaseContainer,
    {$ENDIF}
  {$ENDIF}

  {$IFDEF VSNET}
  Graphics,
  {$ELSE}
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses,
    {$ENDIF}
  {$IFDEF Linux}QControls,{$ELSE}Controls,{$ENDIF}
  {$IFDEF Linux}QForms,{$ELSE}Forms,{$ENDIF}
  {$IFDEF Linux}QGraphics,{$ELSE}Graphics,{$ENDIF}
  {$IFDEF Linux}Qt, Types,{$ELSE}Windows, Messages,{$ENDIF}
  {$ENDIF}
  {$IFDEF CLR}
    IWNetFileParser,
  {$ELSE}
    IWFileParser,
  {$ENDIF}  
  IWContainer, IWControl, IWHTMLTag, IWBaseContainerLayout,
  IWColor, IWBaseInterfaces, IWRenderContext, IWContainerLayout, IWBaseHTMLInterfaces,
  IWBaseHTMLControl, IWGridCommon, IWHTML40Interfaces, IWScriptEvents, IWTypes,
  IWContainerBorderOptions, IWApplication, IWFont, IWBaseControl,
  IWMarkupLanguageTag, IWHTMLContainer, IWBaseRenderContext,
  IWStreams, IWServer;
type
  {$IFDEF VSNET}
  [
   ToolboxItemFilter('TIWAppForm'),
   TIWToolPalette('Intraweb Controls')
  ]
  {$ENDIF}
  TIWCustomRegion = class(TIWHTMLContainer, IIWHTML40Container, IIWBaseComponent, IIWBaseHTMLComponent, IIWBaseControl, IIWBaseHTMLControl, IIWHTML40Control, IIWHTML40Component)
  private
    FBorderOptions: TIWContainerBorderOptions;
    FIWControlImplementation: TIWHTMLControlImplementation;
    FIWBaseControlImplementation: TIWBaseControlImplementation;
    FDoRefreshControl: Boolean;
    FOnCreate: TNotifyEvent;
    FUseFrame: Boolean;
    FRegionDIV: TIWHTMLTag;
    FWebApplication: TIWApplication;

    procedure SetBorderOptions(const Value: TIWContainerBorderOptions);
    procedure OnBorderChange(ASender: TObject);
  {$IFDEF CLR}strict  {$ENDIF} protected
    procedure Dispose(ADispose: Boolean); override;
  protected
    {$IFDEF VSNET}
    procedure AdjustClientRect(var Rect: Rectangle); override;
    function GetClientRect: Rectangle; override;
    {$ELSE}
    procedure AdjustClientRect(var Rect: TRect); override;
    function GetClientRect: TRect; override;
    {$ENDIF}
    function InitPaintHandler: TIWPaintHandler; override;
    procedure Loaded; override;

    // IIWContainer methods
    function InitContainerContext(AWebApplication: TIWApplication): TIWContainerContext; override;

    // IIWControl methods
    function get_HTMLLeft: Integer;
    function get_HTMLTop: Integer;

    function RenderCSSClass(AComponentContext: TIWBaseHTMLComponentContext): string; virtual;
    function RenderStyle(AComponentContext: TIWBaseHTMLComponentContext): string; virtual;
    procedure RenderScripts(AComponentContext: TIWBaseHTMLComponentContext); virtual;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function WantsSubmitValidation: Boolean;

    procedure set_WebFont(AValue: TIWFont);
    function get_WebFont: TIWFont;

    function get_Canvas: TCanvas; override;

    function ParentContainer: IIWBaseContainer;
    procedure PaintTo(ACanvas: TCanvas);

    function SupportsSubmit: Boolean;
    function SupportsInput: Boolean;
    procedure AddFreeNotifier(AObject: TObject);
    procedure RemoveFreeNotifier(AObject: TObject);
    function ComponentContextClass: TIWBaseComponentContextClass; virtual;

    procedure MakeHTMLTag(ATag: TIWHTMLTag; ABuffer: TIWRenderStream);overload;
    function  MakeHTMLTag(ATag: TIWHTMLTag):string;overload; // SLOW, for backward compatibility only
    procedure DoHTMLTag(ATag: TIWHTMLTag);
    procedure  DoCreate;
    //
    procedure InitDesignTime; override;
    procedure InitControl; override;

    {$IFDEF VSNET}
    procedure SetName(const AValue: String); override;
    procedure setDock(AValue: TAlign);
    function getDock: TAlign;
    {$ELSE}
    procedure SetName(const AValue: TComponentName); override;
    {$ENDIF}

    procedure RenderComponents(AContainerContext: TIWContainerContext; APageContext: TIWBasePageContext); override;

    function NeedsFormTag: Boolean; virtual;

    function get_RenderSize: Boolean;
    procedure set_RenderSize(AValue: Boolean);

    function get_ScriptEvents: TIWScriptEvents;
    procedure set_ScriptEvents(Value: TIWScriptEvents);

    function get_Clip: Boolean;
    procedure set_Clip(AValue: Boolean);

    function HintEvents(ATag: TIWHTMLTag{; const AHint: string}): string;
    procedure HookEvents(APageContext: TIWPageContext40;
      AScriptEvents: TIWScriptEvents); {$IFDEF Linux} reintroduce; {$ENDIF} virtual;

    procedure SetActiveControl(AControl: IIWHTML40Control);

    function GenerateControlPositions: String;

    function CheckComponentForRender(AComponent: TPlatformComponent): Boolean; override;
    function DoPostRenderProcessing(ATag: TIWMarkupLanguageTag;
      AComponentContext: TIWBaseComponentContext; AComponent: TPlatformComponent): TIWMarkupLanguageTag; override;

    function CheckUpdateTree(AComponent: TPlatformComponent): Boolean; override;

    // IIWBaseComponent interface
    function get_ComponentName: String;

    function get_SupportsPartial: Boolean;
    procedure set_SupportsPartial(AValue: BOolean);

    function getTabOrder: TIWTabOrder; override;
    procedure setTabOrder(AValue: TIWTabOrder); override;

    {$IFDEF VSNET}
    function ShouldSerializeLayoutMgr: Boolean;
    {$ENDIF}
    //
    property SupportsPartial: Boolean read get_SupportsPartial write set_SupportsPartial;
    property ControlImplementation: TIWHTMLControlImplementation read FIWControlImplementation; // implements IIWBaseComponent, IIWBaseControl, IIWControl;

    // Those were published for some reason - moved them in protected for the custom region, will publish
    // in descendants
    function get_WebCursor: TIWCursor;
    procedure set_WebCursor(AValue: TIWCursor);


    function get_ExtraTagParams: TIWStringList;
    procedure set_ExtraTagParams(const Value: TIWStringList);

    function get_OnHTMLTag: TIWOnHTMLTag;
    procedure set_OnHTMLTag(AValue: TIWOnHTMLTag);

    procedure set_LayoutMgr(Value: TIWContainerLayout);
    function get_LayoutMgr: TIWContainerLayout;

    function get_ZIndex: Integer;
    procedure set_ZIndex(AValue: Integer);

    {$IFNDEF CLR}
    procedure set_WebColor(Value: TIWColor); override;
    function get_WebColor: TIWColor; override;
    {$ENDIF}

    {$IFDEF VSNET}
    [Editor('Atozed.Intraweb.Design.DockEditor, Atozed.Intraweb.Design',
    typeof(UITypeEditor))]
    property Dock: TAlign read getDock write setDock;
    {$ELSE}
    property Align;
    {$ENDIF}
    property Anchors;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    {$IFDEF CLR}
    property WebColor;
    {$ELSE}
    property Color;
    {$ENDIF}
    property ParentShowHint;
    property ShowHint;
    property Hint;

    {$IFDEF CLR}
    {$IFDEF VSNET} [Category('Appearance')] {$ENDIF}
    property WebCursor: TIWCursor read get_WebCursor write set_WebCursor;
    {$ELSE}
    property Cursor: TIWCursor read get_WebCursor write set_WebCursor;
    {$ENDIF}

    {$IFDEF VSNET} [Category('Layout')] {$ENDIF}
    property ClipRegion: Boolean read get_Clip write set_Clip default true;
    {$IFDEF VSNET} [Category('Layout')] {$ENDIF}
    property LayoutMgr: TIWContainerLayout read get_LayoutMgr write set_LayoutMgr;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property ExtraTagParams: TIWStringList read get_ExtraTagParams write set_ExtraTagParams;
    property ZIndex: Integer read get_ZIndex write set_ZIndex;
    //
    property OnHTMLTag: TIWOnHTMLTag read get_OnHTMLTag write set_OnHTMLTag;
    property OnCreate: TNotifyEvent read FOnCreate write FOnCreate;
  public
    {$IFNDEF VSNET}
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure Invalidate; override;
    {$ENDIF}

    function get_DoRefreshControl: Boolean;
    procedure set_DoRefreshControl(AValue: Boolean);

    function get_HTMLName: String;

    function get_UseFrame: Boolean;
    procedure set_UseFrame(AValue: Boolean);

    function get_HTMLWidth: Integer;
    function get_HTMLHeight: Integer;
    //
    {$IFDEF VSNET}[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]{$ENDIF}
    property BorderOptions: TIWContainerBorderOptions read FBorderOptions write SetBorderOptions;
    property DoRefreshControl : Boolean read get_DoRefreshControl write set_DoRefreshControl;
    property HTMLName: string read get_HTMLName;
    property HTMLWidth: Integer read get_HTMLWidth;
    property HTMLHeight: Integer read get_HTMLHeight;
    property UseFrame: Boolean read get_UseFrame write set_UseFrame;
    {$IFNDEF CLR}
    property WebColor;
    {$ENDIF}
  published
    function get_Visible: Boolean; override;
    procedure set_Visible(Value: Boolean); override;

    {$IFDEF CLR}
    // TODO: Remove this when .NET Delphi compiler is fixed
    property Visible read get_Visible write set_Visible;
    {$ELSE}
    property Visible;
    {$ENDIF}
    property TabOrder;
  end;

  {$IFDEF VSNET}
  //TODO: This should be in its own VSNet only unit, but now that we have namespace we are screwed..
  [
    Designer('Atozed.Intraweb.Design.UserControlDesigner, Atozed.Intraweb.Design'
     , typeof(IRootDesigner)),
    DesignerCategory('UserControl')
  ]
  TIWUserControl = class(TIWCustomRegion)
  protected
    function ContainerPrefix: string; override;
    function InitPaintHandler: TIWPaintHandler; override;
    procedure InitControl; override;
    procedure Loaded; override;
    procedure Paint; override;
  published
    {$IFDEF VSNET}
    property Dock;
    {$ELSE}
    property Align;
    {$ENDIF}
    property Anchors;
    property BorderOptions;
    property TabOrder;
    {$IFDEF CLR}
    property WebColor;
    {$ELSE}
    property Color;
    {$ENDIF}
    property ParentShowHint;
    property ShowHint;
    property Hint;

    {$IFDEF CLR}
    {$IFDEF VSNET} [Category('Appearance')] {$ENDIF}
    property WebCursor;
    {$ELSE}
    property Cursor;
    {$ENDIF}

    property ClipRegion;
    property LayoutMgr;
    property ExtraTagParams;
    property ZIndex;
    //
    property OnHTMLTag;
    property OnCreate;
  end;
  {$ENDIF}

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWRegion.bmp}
  TIWRegion = class;
  [
    ToolboxItem(true), ToolboxBitmap(typeof(TIWRegion), 'TIWRegion.bmp'),
    Designer('Atozed.Intraweb.Design.IWContainerDesinger, Atozed.Intraweb.Design', typeof(IDesigner)),
    DesignerCategory('Component')
  ]
  {$ENDIF}
  TIWRegion = class(TIWCustomRegion, IIWInputControl)
  protected
    FSplitter : Boolean;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    procedure HookEvents(APageContext: TIWPageContext40;AScriptEvents: TIWScriptEvents); override;
    procedure InitControl; override;
    function RenderMarkupLanguageTag(AContext: TIWBaseComponentContext): TIWMarkupLanguageTag; override;
    function ContainerPrefix: string; override;


    // from IIWInputControl
    procedure SetValue(const AValue: string);
    procedure SetFileValue(AFile: THTTPFile);
    function get_ControlEncode: TIWControlEncode;
    procedure set_ControlEncode(AValue: TIWControlEncode);
    function get_NonEditableAsLabel: Boolean;
    procedure set_NonEditableAsLabel(AValue: Boolean);
    function get_Editable: Boolean;
    procedure set_Editable(aValue: Boolean);
    procedure GetInputControlNames(ANames: TStringList);
    function IsForThisControl(AName: string): boolean;
  published
    {$IFDEF VSNET}
    property Dock;
    {$ELSE}
    property Align;
    {$ENDIF}
    property Anchors;
    property BorderOptions;
    property TabOrder;
    {$IFDEF CLR}
    property WebColor;
    {$ELSE}
    property Color;
    {$ENDIF}
    property ParentShowHint;
    property ShowHint;
    property Hint;

    {$IFDEF CLR}
    {$IFDEF VSNET} [Category('Appearance')] {$ENDIF}
    property WebCursor;
    {$ELSE}
    property Cursor;
    {$ENDIF}

    property ClipRegion;
    property LayoutMgr;
    property ExtraTagParams;
    property ZIndex;
    //
    property OnHTMLTag;
    property OnCreate;
    //
    property Splitter : Boolean read FSplitter write FSplitter;
  end;

implementation

uses
  IWForm, IWLayoutMgrForm, IWBaseHTMLComponent,
  SWSystem, SysUtils, InGlobal;

{$IFDEF VSNET}
procedure TIWCustomRegion.setDock(AValue: TAlign);
begin
  Align := AValue;
end;

function TIWCustomRegion.getDock: TAlign;
begin
  Result := Align;
end;

function TIWCustomRegion.ShouldSerializeLayoutMgr: Boolean;
begin
  Result := Assigned(FLayoutMgr);
end;
{$ENDIF}

procedure TIWCustomRegion.Loaded;
begin
  inherited Loaded;
  DoCreate;
end;

function TIWCustomRegion.CheckComponentForRender(AComponent: TPlatformComponent): Boolean;
begin
  if SupportsInterface(Parent, IIWHTML40Form) then begin
    Result := HTML40FormInterface(Parent).PageContext.UpdateMode = umPartial;
  end else begin
    if SupportsInterface(Parent, IIWHTML40Container) then begin
      Result := HTML40ContainerInterface(Parent).CheckComponentForRender(AComponent);
    end else begin
      {$IFNDEF VSNET}
      if Parent is TFrame then begin
        if SupportsInterface(Parent.Parent, IIWHTML40Container) then begin
          Result := HTML40ContainerInterface(Parent.Parent).CheckComponentForRender(AComponent);
        end else begin
          Result := false;
        end;
      end else {$ENDIF} begin
        Result := false;
      end;
    end;
  end;
end;

function TIWCustomRegion.DoPostRenderProcessing(ATag: TIWMarkupLanguageTag;
  AComponentContext: TIWBaseComponentContext; AComponent: TPlatformComponent): TIWMarkupLanguageTag;
var
  LIWControl: IIWHTML40Control;
  LTag: TIWHTMLTag;
begin
  Result := ATag;
  LTag := TIWHTMLTag(ATag);
  if SupportsInterface(AComponent, IIWHTML40Control) then begin
    LIWControl := HTML40ControlInterface(AComponent);
    if LIWControl.UseFrame and TIWContainerLayout(ContainerContext.LayoutManager).AllowFrames then begin
      Result := TIWContainerBorderOptions.CreateDefaultFrame(LTag, TIWPageCOntext40(AComponentContext.BasePageContext), LIWControl);
    end;

    AComponentContext.MarkupLanguageTag := Result;

    HTML40ComponentInterface(AComponent).RenderScripts(TIWBaseHTMLComponentContext(AComponentContext));
  end;
end;

function TIWCustomRegion.CheckUpdateTree(AComponent: TPlatformComponent): Boolean;
begin
  Result := false;
  if SupportsInterface(AComponent, IIWHTML40Component) then begin
    Result := HTML40ComponentInterface(AComponent).DoRefreshControl;
  end;
end;

procedure TIWCustomRegion.SetBorderOptions(const Value: TIWContainerBorderOptions);
begin
  FBorderOptions.Assign(Value);
  BorderOptions.Color := FBorderOptions.Color;
  BorderOptions.NumericWidth := FBorderOptions.NumericWidth;
  BorderOptions.Style := FBorderOptions.Style;
  BorderOptions.BorderWidth := FBorderOptions.BorderWidth;
  RequestAlign;
  Invalidate;
end;

procedure TIWCustomRegion.InitControl;
begin
  FIWControlImplementation := TIWHTMLControlImplementation.Create(Self);
  FIWBaseControlImplementation := TIWBaseControlImplementation.Create(Self);

  inherited;

  FBorderOptions := TIWContainerBorderOptions.Create;
  FBorderOptions.Color := fromTColor(clNone);
  FBorderOptions.Style := cbsSolid;
  FBorderOptions.BorderWidth := cbwNumeric;
  FBorderOptions.NumericWidth := 1;

  FBorderOptions.OnChange := OnBorderChange;

  {$IFNDEF VSNET}
  ControlStyle := [csAcceptsControls, csCaptureMouse, csClickEvents, csSetCaption, csDoubleClicks];
  {$ENDIF}
  WebColor := fromTColor(clNone);

  {$IFDEF VSNET}
  FWidth := 60;
  FHeight := 60;
  {$ELSE}
  Width := 60;
  Height := 60;
  {$ENDIF}

  set_ZIndex(1000);
  ClipRegion:= true;
  ShowHint := True;
end;

procedure TIWCustomRegion.InitDesignTime;
begin
  inherited;

  if IsDesignMode then begin
    {$IFNDEF VSNET}
    ControlState := [csCustomPaint];
    {$IFDEF Linux}
    VertScrollBar.Visible := false;
    HorzScrollBar.Visible := false;
    {$ENDIF}
    {$ENDIF}
  end;
end;

function TIWCustomRegion.NeedsFormTag: Boolean;
begin
  Result := false;
end;

function TIWCustomRegion.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  FWebApplication := AContext.WebApplication;
  {$IFDEF VSNET}
  BorderOptions.InitalizeForRendering(WebColor, true, true);
  {$ELSE}
  BorderOptions.InitalizeForRendering(WebColor, HorzScrollBar.Visible, VertScrollBar.Visible);
  {$ENDIF}
  FregionDiv := BorderOptions.CreateFrame(nil, TIWPageContext40(AContext.PageContext), Self);
//  if FWebApplication.IsPartialUpdate then begin
//    FregionDiv.AddStringParam('style', FregionDiv.Params.Values['style']+ 'left: ' + inttostr(Left)+ ';top: ' + inttostr(Top)+';');
//  end;
  Result := FregionDiv;
end;


(*procedure TIWRegion.SetAlignment(const Value: TAlignment);
begin
  FAlignment := Value;
  Invalidate;
end;*)

(*procedure TIWRegion.SetVAlign(const Value: TIWGridVAlign);
begin
  // FVAlign := Value;
  Invalidate;
end;*)

function TIWCustomRegion.get_ExtraTagParams: TIWStringList;
begin
  Result := FIWControlImplementation.get_ExtraTagParams;
end;

procedure TIWCustomRegion.set_ExtraTagParams(const Value: TIWStringList);
begin
  FIWControlImplementation.set_ExtraTagParams(Value);
end;

function TIWCustomRegion.get_OnHTMLTag: TIWOnHTMLTag;
begin
  Result := FIWControlImplementation.OnHTMLTag;
end;                   

procedure TIWCustomRegion.set_OnHTMLTag(AValue: TIWOnHTMLTag);
begin
  FIWControlImplementation.setOnHTMLTag(AValue);
end;

procedure TIWCustomRegion.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FBorderOptions);
  LayoutMgr := nil; // Cleanup layout manager
  inherited;
  if ADispose then begin
    FreeAndNil(FIWControlImplementation);
    FreeAndNil(FIWBaseControlImplementation);
  end;
end;

function TIWCustomRegion.get_HTMLName: String;
begin
  Result := FIWControlImplementation.HTMLName;
end;

function TIWCustomRegion.get_DoRefreshControl: Boolean;
begin
  Result := FDoRefreshControl;
end;

procedure TIWCustomRegion.set_DoRefreshControl(AValue: Boolean);
begin
  FDoRefreshControl := AValue;
end;

function TIWCustomRegion.get_Clip: Boolean;
begin
  Result := FIWControlImplementation.GetClip;
end;

function TIWCustomRegion.get_RenderSize: Boolean;
begin
  Result := True;
end;

function TIWCustomRegion.get_ScriptEvents: TIWScriptEvents;
begin
  Result := FIWControlImplementation.GetScriptEvents;
end;

function TIWCustomRegion.get_ZIndex: Integer;
begin
  Result := FIWControlImplementation.GetZIndex;
end;

procedure TIWCustomRegion.set_Clip(AValue: Boolean);
begin
  FIWControlImplementation.SetClip(AValue);
end;

procedure TIWCustomRegion.set_RenderSize(AValue: Boolean);
begin
  // Constant
end;

procedure TIWCustomRegion.set_ScriptEvents(Value: TIWScriptEvents);
begin
  FIWControlImplementation.SetScriptEvents(Value);
end;

procedure TIWCustomRegion.set_ZIndex(AValue: Integer);
begin
  FIWControlImplementation.SetZIndex(AValue);
end;

function TIWCustomRegion.HintEvents(ATag: TIWHTMLTag{;
  const AHint: string}): string;
begin
  if ShowHint and (Hint <> '') then begin
    ATag.AddStringParam('title', StringReplace(Hint, '''', '&#39', [rfReplaceAll]));
  end;
end;

procedure TIWCustomRegion.HookEvents(
  APageContext: TIWPageContext40;
  AScriptEvents: TIWScriptEvents
  );
begin
end;

procedure TIWCustomRegion.set_Visible(Value: Boolean);
begin
  inherited;
  FIWControlImplementation.SetVisible(Value);
  if Parent is TIWBaseContainer then
  begin
    TIWBaseContainer(Parent).ForceAlign;
  end;
  FDoRefreshControl := true;
end;

procedure TIWCustomRegion.SetActiveControl(AControl: IIWHTML40Control);
begin
  IWHTML40Interfaces.SetActiveControl(Parent, AControl);
end;

procedure TIWCustomRegion.set_LayoutMgr(Value: TIWContainerLayout);
begin
  if Assigned(FLayoutMgr) then begin
    FLayoutMgr.SetContainer(nil);
    {$IFNDEF VSNET}
    FLayoutMgr.RemoveFreeNotification(Self);
    {$ENDIF}
  end;
  FLayoutMgr := Value;
  if Assigned(FLayoutMgr) then begin
    {$IFNDEF VSNET}
    FLayoutMgr.FreeNotification(Self);
    {$ENDIF}
    FLayoutMgr.SetContainer(Self);
  end;
end;

function TIWCustomRegion.get_LayoutMgr: TIWContainerLayout;
begin
  Result := TIWContainerLayout(FLayoutMgr);
end;

procedure TIWCustomRegion.RenderComponents(AContainerContext: TIWContainerContext; APageContext: TIWBasePageContext);
var
  LBuffer: TIWRenderStream;
begin
  ContainerContext := AContainerContext;
  inherited RenderComponents(ContainerContext, APageContext);
  TIWContainerLayout(ContainerContext.LayoutManager).ProcessControls(ContainerContext, TIWBaseHTMLPageContext(APageContext));
  LBuffer := TIWRenderStream.Create;
  try
    with TIWContainerLayout(ContainerContext.LayoutManager) do try
      Process(LBuffer, ContainerContext, APageContext);
      SetContainer(nil);
    finally
      FRegionDiv.Contents.AddBuffer(LBuffer);
    end;
  finally FreeAndNil(LBuffer); end;
end;

function TIWCustomRegion.get_UseFrame: Boolean;
begin
  Result := FUseFrame;
end;

procedure TIWCustomRegion.set_UseFrame(AValue: Boolean);
begin
  FUseFrame := AValue;
end;

procedure TIWCustomRegion.OnBorderChange(ASender: TObject);
var a:integer;
begin
  {BorderOptions.Style := TIWContainerBorderOptions(ASender).Style;
  BorderOptions.NumericWidth := TIWContainerBorderOptions(ASender).NumericWidth;
  BorderOptions.Color := TIWContainerBorderOptions(ASender).Color;
  BorderOptions.Width := TIWContainerBorderOptions(ASender).Width;}
{$IFNDEF VSNET}
  for a:=0 to ControlCount-1 do begin
    if Controls[a].Left<BorderOptions.PixelWidth then begin Controls[a].Left:=BorderOptions.PixelWidth;end;
    if Controls[a].Top<BorderOptions.PixelWidth then begin Controls[a].Top:=BorderOptions.PixelWidth;end;
  end;
{$ENDIF}
  RequestAlign;
  Invalidate;
end;

function TIWCustomRegion.GenerateControlPositions: String;
var
  i: Integer;
begin
  Result := '';
  with HTML40ContainerInterface(Self) do begin
    for i := 0 to IWComponentsCount - 1 do begin
      if SupportsInterface(Component[i], IIWHTML40Control) then begin
        with HTML40ControlInterface(Component[i]) do begin
          if DoRefreshControl then begin
            Result := Result + EOL
              + 'IWTop().' + HTMLName + Format('IWCL.SetRect(new parent.Rect(%d, %d, %d, %d));',
                [HTMLLeft, HTMLTop, HTMLWidth, HTMLHeight]) + EOL;
          end;
        end;
      end;
      if SupportsInterface(Component[i], IIWHTML40Container) then begin
        Result := Result + EOL
          + HTML40ContainerInterface(Component[i]).GenerateControlPositions;
      end;
    end;
  end;
end;

{$IFNDEF VSNET}
procedure TIWCustomRegion.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  if Parent is TIWCustomRegion then begin
    if TIWCustomRegion(Parent).BorderOptions.PixelWidth>ALeft then begin
      ALeft:=TIWCustomRegion(Parent).BorderOptions.PixelWidth;
    end;
  end;
  if Parent is TIWCustomRegion then begin
    if TIWCustomRegion(Parent).BorderOptions.PixelWidth>ATop then begin
      ATop:=TIWCustomRegion(Parent).BorderOptions.PixelWidth;
    end;
  end;
  DoRefreshControl := (ALeft <> Left) or (ATop <> Top) or (AWidth <> Width) or (AHeight <> Height) or DoRefreshControl;
  inherited;
end;
{$ENDIF}

{$IFDEF VSNET}
procedure TIWCustomRegion.SetName(const AValue: String);
{$ELSE}
procedure TIWCustomRegion.SetName(const AValue: TComponentName);
{$ENDIF}
begin
  if Name <> 'IWFrameRegion' then begin
    inherited;
  end else begin
    Align := alClient;
  end;
end;

function TIWCustomRegion.ComponentContextClass: TIWBaseComponentContextClass;
begin
  Result := TIWComponent40Context;
end;

// IIWContainer methods

function TIWCustomRegion.InitContainerContext(AWebApplication: TIWApplication): TIWContainerContext;
var
  LLayoutMgr: TIWBaseContainerLayout;
begin
  Result := IWBaseRenderContext.TIWContainerContext.Create(AWebApplication);
  LLayoutMgr := LayoutMgr;

  if LLayoutMgr = nil then begin
    LLayoutMgr := TIWLayoutMgrForm.Create(Self);
  end else begin
    LLayoutMgr.SetContainer(Self);
    if not LLayoutMgr.Able then begin
      LLayoutMgr := TIWLayoutMgrForm.Create(Self);
    end;
  end;

  LLayoutMgr.SetContainer(Self);
  Result.LayoutManager := LLayoutMgr;
end;

procedure TIWCustomRegion.DoCreate;
begin
  if Assigned(OnCreate) and not IsDesignMode then begin
    OnCreate(Self);
  end;
end;

procedure TIWCustomRegion.DoHTMLTag(ATag: TIWHTMLTag);
begin
  FIWControlImplementation.DoHTMLTag(ATag);
end;

function TIWCustomRegion.get_Canvas: TCanvas;
begin
  Result := FIWBaseControlImplementation.GetCanvas;
end;

function TIWCustomRegion.get_WebCursor: TIWCursor;
begin
  Result := FIWControlImplementation.GetCursor;
end;

function TIWCustomRegion.get_WebFont: TIWFont;
begin
  Result := FIWControlImplementation.Font;
end;

function TIWCustomRegion.get_HTMLHeight: Integer;
begin
  if IsDesignMode then begin
    Result := FIWControlImplementation.HTMLHeight;
  end else begin
    Result := FIWControlImplementation.HTMLHeight;
   if Assigned(FWebApplication) then
   begin { Hidden region have FWebApplication = nil but are resized at list if they are inside another region }
     Result := Result - SWSystem.iif(FWebApplication.Browser in [brNetscape6, brNetscape7], FBorderOptions.PixelWidth * 2, 0);
   end;
  end;
end;

function TIWCustomRegion.get_HTMLLeft: Integer;
begin
  Result := FIWControlImplementation.GetHTMLLeft;
  {$IFDEF VSNET}
  {$ELSE}
  if Parent is TFrame then begin
    Result := Parent.Left;
  end;
  {$ENDIF}
end;

function TIWCustomRegion.get_HTMLTop: Integer;
begin
  Result := FIWControlImplementation.GetHTMLTop;
  {$IFDEF VSNET}
  {$ELSE}
  if Parent is TFrame then begin
    Result := Parent.Top;
  end;
  {$ENDIF}
end;

function TIWCustomRegion.get_HTMLWidth: Integer;
begin
  if IsDesignMode then begin
    Result := FIWControlImplementation.HTMLWidth;
  end else begin
    Result := FIWControlImplementation.HTMLWidth;
    if Assigned(FWebApplication) then begin { Hidden region have FWebApplication = nil
       but are resized at list if they are inside another region }
       Result := Result -
         SWSystem.iif(FWebApplication.Browser in [brNetscape6, brNetscape7], FBorderOptions.PixelWidth * 2, 0);
    end;
  end;
end;

{$IFNDEF CLR}
function TIWCustomRegion.get_WebColor: TIWColor;
begin
  Result := FIWControlImplementation.GetIWColor;
end;

procedure TIWCustomRegion.set_WebColor(Value: TIWColor);
begin
//  BorderOptions.BGColor := Value;
  FIWControlImplementation.SetIWColor(Value);
end;
{$ENDIF}

function TIWCustomRegion.getTabOrder: TIWTabOrder;
begin
  {$IFNDEF VSNET}
  if Parent is tframe then
    result := inherited getTabOrder
  else
  {$ENDIF}
    Result := FIWControlImplementation.getTabOrder;
end;

procedure TIWCustomRegion.setTabOrder(AValue: TIWTabOrder);
begin
  {$IFNDEF VSNET}
  if Parent is tframe then begin
    if OwnerForm<>nil then
      FreeNotification(OwnerForm);
    inherited setTabOrder(AValue);
  end else
  {$ENDIF}
    FIWControlImplementation.SetTabOrder(AValue);
end;

{$IFNDEF VSNET}
procedure TIWCustomRegion.Invalidate;
begin
  inherited;
  FDoRefreshControl := true;
end;
{$ENDIF}

procedure TIWCustomRegion.MakeHTMLTag(ATag: TIWHTMLTag; ABuffer: TIWRenderStream);
begin
  FIWControlImplementation.MakeHTMLTag(ATag, ABuffer);
  ABuffer.WriteString(ParentContainer.ContainerContext.ComponentContext[Name].PostRender);
end;

// SLOW, for backward compatibility only
function TIWCustomRegion.MakeHTMLTag(ATag: TIWHTMLTag):string;
var
  AStream: TIWRenderStream;
begin
  AStream := TIWRenderStream.Create;
  try
    MakeHTMLTag(ATag, AStream);
    Result := AStream.Extract;
  finally
    AStream.Free;
  end;
end;

procedure TIWCustomRegion.PaintTo(ACanvas: TCanvas);
{$IFDEF VSNET}
var
  i: Integer;
  Bmp: TBitmap;
  LX, LY: Integer;
{$ENDIF}
begin
  FIWBaseControlImplementation.PaintTo(ACanvas);
  {$IFDEF VSNET}
  if not Assigned(Site) and IsDesignMode then begin
    for i := 0 to IWComponentsCount - 1 do begin
      if SupportsInterface(Component[i], IIWBaseControl) then begin
        Bmp := TBitmap.Create; try
          LX := 0;
          LY := 0;
          if Component[i] is TIWNETBaseControl then begin
            with TIWNETBaseControl(Component[i]) do begin
              Bmp.Width := Width;
              Bmp.Height := Height;

              LX := Left;
              LY := Top;
            end;
          end;
          BaseControlInterface(Component[i]).PaintTo(Bmp.Canvas);

          ACanvas.Draw(LX, LY, Bmp);
        finally
          Bmp.Free;
        end;
      end;
    end;
  end;
  {$ENDIF}
end;

function TIWCustomRegion.ParentContainer: IIWBaseContainer;
begin
  Result := FIWBaseControlImplementation.ParentContainer;
end;

function TIWCustomRegion.RenderCSSClass(
  AComponentContext: TIWBaseHTMLComponentContext): string;
begin
  Result := FIWControlImplementation.RenderCSSClass(AComponentContext);
end;

{$IFDEF VSNET}
procedure TIWCustomRegion.AdjustClientRect(var Rect: Rectangle);
begin
  inherited AdjustClientRect(Rect);

  if IsDesignMode then begin
    Rect.Inflate(-FBorderOptions.PixelWidth, -FBorderOptions.PixelWidth);
  end;
end;

function TIWCustomRegion.GetClientRect: Rectangle;
begin
  Result := inherited GetClientRect;

  if not IsDesignMode then begin
    Result.Inflate(-FBorderOptions.PixelWidth, -FBorderOptions.PixelWidth);
  end;
end;

{$ELSE}
function TIWCustomRegion.GetClientRect: TRect;
begin
  Result := inherited GetClientRect;

  if not IsDesignMode then begin
    InflateRect(Result, -FBorderOptions.PixelWidth, -FBorderOptions.PixelWidth);
  end;
end;

procedure TIWCustomRegion.AdjustClientRect(var Rect: TRect);
begin
  inherited AdjustClientRect(Rect);
  if IsDesignMode then begin
    InflateRect(Rect, -FBorderOptions.PixelWidth, -FBorderOptions.PixelWidth);
  end;
end;
{$ENDIF}

procedure TIWCustomRegion.RenderScripts(
  AComponentContext: TIWBaseHTMLComponentContext);
begin
  FIWControlImplementation.RenderScripts(AComponentContext);
  (* Commenting this fixes BBG694. Rendering the border size via JS clipped the border in Gecko engines
  if AComponentContext.ContainerContext.LayoutManager is TIWLayoutMgrForm then begin
    with TIWPageContext40(AComponentContext.PageContext) do begin
      // if (WebApplication.Browser <> brOpera) then begin
        AddToIWCLInitProc('if (' + HTMLName + 'IWCL) {');
          AddToIWCLInitProc(HTMLName + 'IWCL.BorderWidth = ' + IntToStr(FBorderOptions.PixelWidth) + ';');
        AddToIWCLInitProc('}');
      // end;
    end;
  end;
  *)

  if AComponentContext.ContainerContext.LayoutManager is TIWLayoutMgrForm then begin
    with TIWPageContext40(AComponentContext.PageContext) do begin
      AddToIWCLInitProc('if (' + HTMLName + 'IWCL) {');
        AddToIWCLInitProc(HTMLName + 'IWCL.BorderWidthPixels = ' + IntToStr(FBorderOptions.PixelWidth) + ';');
      AddToIWCLInitProc('}');
    end;
  end;
end;

function TIWCustomRegion.RenderStyle(
  AComponentContext: TIWBaseHTMLComponentContext): string;
begin
  Result := FIWControlImplementation.RenderStyle(AComponentContext);
end;

procedure TIWCustomRegion.set_WebCursor(AValue: TIWCursor);
begin
  FIWControlImplementation.SetCursor(AValue);
end;

procedure TIWCustomRegion.set_WebFont(AValue: TIWFont);
begin
  FIWControlImplementation.Font := AValue;
end;

function TIWCustomRegion.SupportsInput: Boolean;
begin
  Result := FIWControlImplementation.SupportsInput;
end;

function TIWCustomRegion.SupportsSubmit: Boolean;
begin
  Result := FIWControlImplementation.SupportsSubmit;
end;

function TIWCustomRegion.WantsSubmitValidation: Boolean;
begin
  Result := FIWControlImplementation.WantsSubmitValidation;
end;

function TIWCustomRegion.get_Visible: Boolean;
begin
  Result := FIWControlImplementation.GetVisible;
end;

function TIWCustomRegion.InitPaintHandler: TIWPaintHandler;
begin
  Result := TIWBaseControl.CreatePaintHandler(ClassName, Self);
end;

function TIWCustomRegion.get_ComponentName: String;
begin
  Result := HTMLName;
end;

procedure TIWCustomRegion.AddFreeNotifier(AObject: TObject);
begin
end;

procedure TIWCustomRegion.RemoveFreeNotifier(AObject: TObject);
begin
end;

function TIWCustomRegion.get_SupportsPartial: Boolean;
begin
  Result := FIWControlImplementation.get_SupportsPartial;
end;

procedure TIWCustomRegion.set_SupportsPartial(AValue: BOolean);
begin
  FIWControlImplementation.set_SupportsPartial(AValue);
end;

{$IFDEF VSNET}
{ TIWUserControl }

procedure TIWUserControl.InitControl;
begin
  inherited;
  // Inc(GIsLoading);
end;

function TIWUserControl.ContainerPrefix: string;
begin
  Result := HTMLName;
end;

function TIWUserControl.InitPaintHandler: TIWPaintHandler;
begin
  // Use IWRegion paint handler
  Result := TIWBaseControl.CreatePaintHandler('TIWRegion', Self);
end;

procedure TIWUserControl.Loaded;
begin
  inherited;
  // do not invoke OnCreate at design-time
  // this will make a user's runtime code to execute
  if Assigned(FOnCreate) and (not IsDesignMode) then begin
    FOnCreate(Self);
  end;
end;

procedure TIWUserControl.Paint;
var
  i: Integer;
  Bmp: TBitmap;
  LX, LY: Integer;
begin
  inherited;
  for i := 0 to IWComponentsCount - 1 do begin
    if SupportsInterface(Component[i], IIWBaseControl) then begin
      Bmp := TBitmap.Create; try
        LX := 0;
        LY := 0;
        if Component[i] is TIWNETBaseControl then begin
          with TIWNETBaseControl(Component[i]) do begin
            Bmp.Width := Width;
            Bmp.Height := Height;

            LX := Left;
            LY := Top;
          end;
        end;
        BaseControlInterface(Component[i]).PaintTo(Bmp.Canvas);

        Canvas.Draw(LX, LY, Bmp);
      finally
        Bmp.Free;
      end;
    end;
  end;
end;
{$ENDIF}

{ TIWRegion }

function TIWRegion.ContainerPrefix: string;
begin
  {$IFNDEF VSNET}
  if Owner is TFrame then begin
    Result := UpperCase(TFrame(Owner).Name);
  end else
  {$ENDIF}
    if SupportsInterface(Parent, IIWBaseContainer) then begin
      Result := BaseContainerInterface(Parent).ContainerPrefix;
    end else begin
      Result := UpperCase(Name);
    end;
end;

function TIWRegion.RenderMarkupLanguageTag(AContext: TIWBaseComponentContext): TIWMarkupLanguageTag;
begin
  result := RenderHTML(TIWBaseHTMLComponentContext(AContext));
  if GGetWebApplicationThreadVar.IsPartialUpdate and Assigned(result) then begin
    result.AddStringParam('style', result.Params.Values['style']+ 'left: ' + inttostr(Left)+ ';top: ' + inttostr(Top) + ';width: ' + IntToStr(HTMLWidth) + ';height: ' + IntToStr(HTMLHeight) + ';');
  end;
end;

procedure TIWRegion.GetInputControlNames(ANames: TStringList);
begin
  if (FSplitter) and (Align in [alLeft,alRight,alTop,alBottom]) then begin
    ANames.Add(HTMLName + '_data');
  end;
end;

procedure TIWRegion.SetFileValue(AFile: THTTPFile);
begin
  // empty
end;

function TIWRegion.get_ControlEncode: TIWControlEncode;
begin
  result := ceNone;
end;

procedure TIWRegion.SetValue(const AValue: string);
var
  LVal, s : string;
  LLocation, LSize : Integer;
begin
  if (FSplitter) and (Align in [alLeft,alRight,alTop,alBottom]) and (AValue <> '') then begin
    LVal := AValue;
    s := Fetch(LVal, ',');
    LLocation := StrToInt(s);
    LSize := StrToInt(LVal);

    case Align of
      alTop: begin
        Height := LSize;
      end;

      alBottom: begin
        Height := LSize;
        Top := LLocation;
      end;

      alLeft: begin
        Width := LSize;
      end;
      
      alRight: begin
        Left := LLocation;
        Width := LSize;
      end;
    end;
    TIWBaseContainer(Parent).ForceAlign;
  end;
end;

procedure TIWRegion.set_ControlEncode(AValue: TIWControlEncode);
begin
  // empty
end;

function TIWRegion.get_NonEditableAsLabel: Boolean;
begin
  result := false;
end;

procedure TIWRegion.set_NonEditableAsLabel(AValue: Boolean);
begin
  // empty
end;

function TIWRegion.get_Editable: Boolean;
begin
  result := True;
end;

procedure TIWRegion.set_Editable(aValue: Boolean);
begin
// empty
end;

function TIWRegion.IsForThisControl(AName: string): boolean;
begin
  Result := AnsiSameText(AName, HTMLName + '_data');
end;

procedure TIWRegion.HookEvents(APageContext: TIWPageContext40;AScriptEvents: TIWScriptEvents);
//var
//  LGripType : string;
begin
{
  if (FSplitter) and (Align in [alLeft,alRight,alTop,alBottom]) then begin

    case Align of
      alRight   : LGripType := 'left';
      alBottom  : LGripType := 'top';
      alLeft    : LGripType := 'right';
      alTop     : LGripType := 'bottom';
    end;

    AScriptEvents.HookEvent('onmousedown','return SplitterMouseDown(event,''' + LGripType + ''')');
    AScriptEvents.HookEvent('onmousemove','return SplitterMouseMove(event,''' + LGripType + ''')');
  end;
}
end;

function TIWRegion.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LSplitterSize, LSplitterLocation : Integer;
  LGripType : string;
  LGripStyle : string;
  LIsW3C : Boolean;
  LGripWidth, LGripHeight, LGripTop : string;
  LFormTag : TIWHTMLTag;
  i : Integer;
  LGripZIndex : Integer;
begin
  Result := inherited RenderHTML(AContext);

  if (FSplitter) and (Align in [alLeft,alRight,alTop,alBottom]) then begin

    //Todo: Check how to do that in VS. ControlCount and Controls is not implemented there
    LGripZIndex := 11000;
    {$IFNDEF VSNET}
    for i := 0 to ControlCount - 1 do begin
      if Controls[i] is TIWCustomControl then begin
        if TIWCustomControl(Controls[i]).ZIndex > LGripZIndex then begin
          LGripZIndex := TIWCustomControl(Controls[i]).ZIndex + 1;
        end;
      end;
    end;
    {$ENDIF$}

    if( Align in [alTop,alBottom] ) then begin
      LSplitterSize := Height;
      LSplitterLocation := Top;
    end else begin
      LSplitterSize := Width;
      LSplitterLocation := Left;
    end;

    LIsW3C := AContext.Browser in [brNetscape6, brNetscape7, brSafari];

    case Align of
      alRight   : LGripType := 'left';
      alBottom  : LGripType := 'top';
      alLeft    : LGripType := 'right';
      alTop     : LGripType := 'bottom';
    end;

    // render Hidden Input

    // !!!surround hidden input with FORM tag cause the region itself is not placed
    // in FORM tag (to avoid FORM tags nesting)
    // Look at IWLayoutMgrForm for explicit IWRegion and IWTabControl support
    LFormTag := Result.Contents.AddTag('FORM');
    LFormTag.AddStringParam('onSubmit', 'return FormDefaultSubmit();');
    with LFormTag.Contents.AddTag('INPUT') do begin
      AddStringParam('name',HTMLName + '_data');
      AddStringParam('id',HTMLName + '_data');
      AddStringParam('type','hidden');
      AddStringParam('value',IntToStr(LSplitterLocation) + ',' + IntToStr(LSplitterSize) );
    end;

    case Align of

      alBottom  : begin
          if LIsW3C then begin
            LGripHeight := '0px';
          end else begin
            LGripHeight := '6px';
          end;
          LGripStyle := 'z-index:' + IntToStr(LGripZIndex) + ';top:0px; WIDTH: 100%; height:' + LGripHeight + ';'
                  + 'border-top: 3px outset; border-bottom: 3px outset;  POSITION: absolute; overflow: hidden;cursor: n-resize;' ;
        end;

      alTop     : begin
          if LIsW3C then begin
            LGripHeight := '0px';
          end else begin
            LGripHeight := '6px';
          end;
          LGripStyle := 'z-index:' + IntToStr(LGripZIndex) + ';bottom:0px; WIDTH: 100%; height:' + LGripHeight + ';'
                  + 'border-top: 3px groove; border-bottom: 3px groove;  POSITION: absolute; overflow: hidden;cursor: n-resize;' ;
        end;

      alRight   : begin
          if LIsW3C then begin
            LGripWidth := '0px';
            LGripTop := '0px';
          end else begin
            LGripWidth := '6px';
            LGripTop := '0px';
          end;
          LGripStyle := 'z-index:' + IntToStr(LGripZIndex) + ';left:0px; top:' + LGripTop + '; height: 100%; width:' + LGripWidth + ';'
                  + 'border-top: none; border-bottom: none;border-left: 3px groove; border-right: 3px groove; POSITION: absolute; overflow: hidden; cursor: w-resize;z-index:10000;' ;
        end;

      alLeft    : begin
          if LIsW3C then begin
            LGripWidth := '0px';
            LGripTop := '0px';
          end else begin
            LGripWidth := '6px';
            LGripTop := '0px';
          end;
          LGripStyle := 'z-index:' + IntToStr(LGripZIndex) + ';right:0px; top: ' + LGripTop + '; height: 100%; width:' + LGripWidth + ';'
                  + 'border-top: none; border-bottom: none;border-left: 3px groove; border-right: 3px groove; POSITION: absolute; overflow: hidden; cursor: w-resize;z-index:10000;' ;
        end;
    end;


    // render the grip
    with Result.Contents.AddTag('DIV') do begin
      AddStringParam('id', HTMLName + '_grip');
      AddStringParam('onmousedown','return GripMouseDown(event,''' + LGripType + ''')');
      AddStringParam('style','' + LGripStyle);
    end;


    // Load the required javascript functions for splitter
    (AContext as TIWComponent40Context).AddScriptFile('/js/IWSplitter.js');
  end;
end;

procedure TIWRegion.InitControl;
begin
  inherited;
  FSplitter := False;
end;

initialization
  TIWServer.AddInternalFile('IW_JS_IWSPLITTER', '/js/IWSplitter.js');

end.

